<?php
/**
 * Mismo: Default Tabbed
 *
 * Widget to display latest posts and comment in tabbed layout.
 *
 * @package Mismo
 * 
 * @since 1.0.0
 */

class Mismo_Default_Tabbed extends WP_widget {

    /**
     * Register widget with WordPress.
     */
    public function __construct() {
        $widget_ops = array( 
            'classname'     => 'mismo_default_tabbed',
            'description'   => esc_html__( 'A widget shows recent posts and comments in tabbed layout.', 'mismo' ),
            'customize_selective_refresh'   => true,
        );
        parent::__construct( 'mismo_default_tabbed', esc_html__( 'Mismo: Default Tabbed', 'mismo' ), $widget_ops );
    }

    /**
     * Helper function that holds widget fields
     * Array is used in update and form functions
     */
    private function widget_fields() {
        
        $fields = array(

            'latest_tab_title' => array(
                'mismo_widgets_name'         => 'latest_tab_title',
                'mismo_widgets_title'        => esc_html__( 'Latest Tab title', 'mismo' ),
                'mismo_widgets_default'      => esc_html__( 'Latest', 'mismo' ),
                'mismo_widgets_field_type'   => 'text'
            ),

            'comments_tab_title' => array(
                'mismo_widgets_name'         => 'comments_tab_title',
                'mismo_widgets_title'        => esc_html__( 'Comments Tab title', 'mismo' ),
                'mismo_widgets_default'      => esc_html__( 'Comments', 'mismo' ),
                'mismo_widgets_field_type'   => 'text'
            ),
            
            'mismo_posts_count' => array(
                'mismo_widgets_name'         => 'mismo_posts_count',
                'mismo_widgets_title'        => esc_html__( 'No. of Posts', 'mismo' ),
                'mismo_widgets_default'      => '5',
                'mismo_widgets_field_type'   => 'number'
            )
        );
        return $fields;
    }

    /**
     * Front-end display of widget.
     *
     * @see WP_Widget::widget()
     *
     * @param array $args     Widget arguments.
     * @param array $instance Saved values from database.
     */
    public function widget( $args, $instance ) {
        extract( $args );

        if ( empty( $instance ) ) {
            return ;
        }

        $mismo_latest_title    = empty( $instance['latest_tab_title'] ) ? esc_html__( 'Latest', 'mismo' ) : $instance['latest_tab_title'];
        $mismo_comments_title  = empty( $instance['comments_tab_title'] ) ? esc_html__( 'Comments', 'mismo' ) : $instance['comments_tab_title'];
        $mismo_post_count      = empty( $instance['mismo_posts_count'] ) ? 5 : $instance['mismo_posts_count'];

        echo $before_widget;
    ?>
            <div class="m-default-tabbed-wrapper" id="m-tabbed-widget">
                
                <ul class="widget-tabs m-clearfix" id="m-widget-tab">
                    <li><a href="#latest"><?php echo esc_html( $mismo_latest_title ); ?></a></li>
                    <li><a href="#comments"><?php echo esc_html( $mismo_comments_title ); ?></a></li>
                </ul><!-- .widget-tabs -->

                <div id="latest" class="m-tabbed-section">
                    <?php
                        $latest_args = array(
                            'posts_per_page' => absint( $mismo_post_count )
                        );
                        $latest_query = new WP_Query( $latest_args );
                        if ( $latest_query->have_posts() ) {
                            while ( $latest_query->have_posts() ) {
                                $latest_query->the_post();
                                if ( has_post_thumbnail() ) {
                                    $thumbail_class = 'mt-clearfix ';
                                } else {
                                    $thumbail_class = 'mt-clearfix no-thumbnail';
                                }
                    ?>
                                <div class="m-single-post <?php echo esc_attr( $thumbail_class ); ?>">
                                    <div class="post-thumb">
                                        <a href="<?php the_permalink(); ?>">
                                            <figure><?php the_post_thumbnail( 'thumbnail' ); ?></figure>
                                        </a>
                                    </div><!-- .m-post-thumb -->
                                    <div class="post-content">
                                        <h3 class="post-title small-size"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
                                        <div class="post-meta">
                                            <?php mismo_posted_on(); ?>
                                        </div>
                                    </div><!-- .m-post-content -->
                                </div><!-- .m-single-post -->
                    <?php
                            }
                        }
                        wp_reset_postdata();
                    ?>
                </div><!-- #latest -->

                <div id="comments" class="m-tabbed-section m-clearfix">
                    <ul>
                        <?php
                            $mismo_comments_count = apply_filters( 'mismo_comment_tabbed_posts_count', 5 );
                            $mismo_tabbed_comments = get_comments( array( 'number' => $mismo_comments_count ) );
                            foreach ( $mismo_tabbed_comments as $comment  ) {
                        ?>
                                <li class="m-single-comment">
                                    <?php
                                        $title = '"'. get_the_title( $comment->comment_post_ID ) .'" - ';
                                        echo '<div class="m-comment-avatar">'. get_avatar( $comment, '120' ) .'</div>';
                                    ?>
                                    <div class="m-comment-desc-wrap">
                                        <strong><?php echo esc_html( strip_tags( $comment->comment_author ) ); ?></strong>
                                        <?php esc_html_e( '&nbsp;commented on', 'mismo' ); ?> 
                                        <a href="<?php echo esc_url( get_permalink( $comment->comment_post_ID ) ); ?>" rel="external nofollow" title="<?php echo esc_attr( $title ); ?>"> <?php echo esc_html( $title); ?></a> <?php echo wp_html_excerpt( $comment->comment_content, 20, ' ...' ); ?>
                                    </div><!-- .m-comment-desc-wrap -->
                                </li>
                        <?php
                            }
                        ?>
                    </ul>
                </div><!-- #comments -->

            </div><!-- .m-default-tabbed-wrapper -->
    <?php
        echo $after_widget;
    }

    /**
     * Sanitize widget form values as they are saved.
     *
     * @see WP_Widget::update()
     *
     * @param   array   $new_instance   Values just sent to be saved.
     * @param   array   $old_instance   Previously saved values from database.
     *
     * @uses    mismo_widgets_updated_field_value()     defined in m-widget-fields.php
     *
     * @return  array Updated safe values to be saved.
     */
    public function update( $new_instance, $old_instance ) {
        $instance = $old_instance;

        $widget_fields = $this->widget_fields();

        // Loop through fields
        foreach ( $widget_fields as $widget_field ) {

            extract( $widget_field );

            // Use helper function to get updated field values
            $instance[$mismo_widgets_name] = mismo_widgets_updated_field_value( $widget_field, $new_instance[$mismo_widgets_name] );
        }

        return $instance;
    }

    /**
     * Back-end widget form.
     *
     * @see WP_Widget::form()
     *
     * @param   array $instance Previously saved values from database.
     *
     * @uses    mismo_widgets_show_widget_field()       defined in m-widget-fields.php
     */
    public function form( $instance ) {
        $widget_fields = $this->widget_fields();

        // Loop through fields
        foreach ( $widget_fields as $widget_field ) {

            // Make array elements available as variables
            extract( $widget_field );
            $mismo_widgets_field_value = ! empty( $instance[$mismo_widgets_name] ) ? wp_kses_post( $instance[$mismo_widgets_name] ) : '';
            mismo_widgets_show_widget_field( $this, $widget_field, $mismo_widgets_field_value );
        }
    }
}